<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class FormatNumbersMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        /** @var \Symfony\Component\HttpFoundation\Response $response */
        $response = $next($request);

        // Only touch JSON responses
        $contentType = (string) $response->headers->get('Content-Type');
        $isJson = $response instanceof JsonResponse
            || Str::startsWith(strtolower($contentType), 'application/json');

        if (! $isJson) {
            return $response;
        }

        // Handle Illuminate\Http\JsonResponse cleanly
        if ($response instanceof JsonResponse) {
            $data = $response->getData(true); // assoc array
            $this->formatNumbers($data);
            $response->setData($data);
            return $response;
        }

        // Fallback for plain Symfony/Laravel Response with JSON body
        $raw = $response->getContent();
        $data = json_decode($raw, true);

        if (json_last_error() === JSON_ERROR_NONE && $data !== null) {
            $this->formatNumbers($data);
            // No JSON_PRESERVE_ZERO_FRACTION — we want 123.00 -> 123
            $response->setContent(json_encode($data, JSON_UNESCAPED_UNICODE));
        }

        return $response;
    }

    /**
     * Recursively walk the array and convert floats with .00 to ints.
     * Keeps real decimals (e.g., 12.5 stays 12.5).
     */
    private function formatNumbers(&$data): void
    {
        if (is_array($data)) {
            foreach ($data as &$v) {
                $this->formatNumbers($v);
            }
            unset($v);
            return;
        }

        // Only act on real numbers (not numeric strings to avoid touching IDs/phones)
        if (is_float($data)) {
            if (floor($data) == $data) {
                $data = (int) $data; // strip .00
            } else {
                // normalize float to drop trailing zeros like 1.2300 -> 1.23 (JSON will handle)
                // leave as float
            }
        }
        // ints are already fine; strings/objects are left untouched
    }
}

